<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * Let's Review Class
 *
 * @since      1.0.0
 *
 * @package    Let's Review
 * @subpackage lets-review/inc
 */

class Lets_Review {
	public $name;
	public $version;
	public $slug;
	public $url;
	public $dir_path;

	/**
	* Constructor
	*
	* @since 1.0.0
	*/
	public function __construct() {
		$this->name     = 'Lets Review';
		$this->version  = '3.4.3';
		$this->slug     = 'lets-review';
		$this->url      = plugin_dir_url( __DIR__ );
		$this->dir_path = plugin_dir_path( __DIR__ );
		$this->lets_review_loader();
		$this->lets_review_locale();
		add_action( 'widgets_init', array( $this, 'lets_review_widget' ) );
		add_action( 'after_setup_theme', array( $this, 'thumbnails' ) );
		$this->lets_review_frontend();
		if ( is_admin() ) {
			$this->lets_review_backend();
			add_action( 'load-post.php', array( $this, 'lets_review_metabox_init' ) );
			add_action( 'load-post-new.php', array( $this, 'lets_review_metabox_init' ) );
		}
		$this->lets_review_blocks();
	}

	/**
	 * Let's Review Loader
	 *
	 * @since 1.0.0
	 */
	private function lets_review_loader() {
		require_once $this->dir_path . 'inc/lets-review-sanitizers.php';
		require_once $this->dir_path . 'admin/class-lets-review-helpers.php';
		require_once $this->dir_path . 'admin/class-lets-review-options.php';
		require_once $this->dir_path . 'admin/class-lets-review-admin.php';
		require_once $this->dir_path . 'admin/class-lets-review-api.php';
		require_once $this->dir_path . 'admin/class-lets-review-button.php';
		require_once $this->dir_path . 'admin/class-lets-review-metabox.php';
		require_once $this->dir_path . 'admin/class-lets-review-settings.php';
		require_once $this->dir_path . 'admin/class-lets-review-blocks.php';
		require_once $this->dir_path . 'admin/class-lets-review-routes.php';
		require_once $this->dir_path . 'admin/lets-review-routes.php';
		require_once $this->dir_path . 'frontend/class-lets-review-frontend.php';
		require_once $this->dir_path . 'frontend/class-lets-review-shortcodes.php';
		require_once $this->dir_path . 'frontend/class-lets-review-visitors.php';
		require_once $this->dir_path . 'inc/class-lets-review-amp.php';
		require_once $this->dir_path . 'inc/class-lets-review-i18n.php';
		require_once $this->dir_path . 'inc/lets-review-widget.php';
	}

	/**
	 * Let's Review Initialize
	 *
	 * @since    1.0.0
	 */
	function lets_review_metabox_init() {
		new Lets_Review_Metabox( $this->slug, $this->version, $this->url );
	}

	/**
	 * Let's Review Widget
	 *
	 * @since 1.0.0
	 */
	public function lets_review_widget() {
		return register_widget( 'Lets_Review_Widget' );
	}

	/**
	 * Let's Review Backend Loader
	 *
	 * @since    1.0.0
	 */
	private function lets_review_backend() {
		new Lets_Review_Admin( $this->slug, $this->version, $this->url );
		new Lets_Review_Settings( $this->slug, $this->version, $this->url, $this->name );
		$button = new Lets_Review_Button( $this->slug, $this->url );
		add_action( 'admin_init', array( $button, 'lets_review_button_init' ) );
	}

	/**
	 * Let's Review Frontend Loader
	 *
	 * @since    1.0.0
	 */
	private function lets_review_frontend() {
		$frontend = new Lets_Review_Frontend( $this->slug, $this->version, $this->url );
		add_filter( 'the_content', array( $frontend, 'lets_review_append' ) );
		add_action( 'wp_footer', array( $this, 'el_check' ) );
		add_action( 'wp_enqueue_scripts', array( $frontend, 'lets_review_enqueue_assets' ) );
		$shortcodes = new Lets_Review_Shortcodes( $this->slug, $this->version, $this->url );
		add_shortcode( 'letsreview', array( $shortcodes, 'lets_review_shortcode' ) );
		add_shortcode( 'letsreviewlist', array( $shortcodes, 'lets_review_shortcode_list' ) );
		add_shortcode( 'letsreviewaffiliate', array( $shortcodes, 'lets_review_shortcode_affiliate' ) );
		add_shortcode( 'letsreviewunique', array( $shortcodes, 'lets_review_shortcode_unique' ) );
		add_shortcode( 'letsreviewuserrating', array( $shortcodes, 'lets_review_shortcode_user_rating' ) );
		add_shortcode( 'letsreviewcommentaveragefinalscore', array( $shortcodes, 'lets_review_shortcode_comment_averages_final_score' ) );
		add_shortcode( 'letsreviewcommentaverage', array( $shortcodes, 'lets_review_shortcode_comment_averages' ) );
		$visitors = new Lets_Review_Visitors( $this->slug, $this->version, $this->url );
		add_action( 'wp_insert_comment', array( $visitors, 'lets_review_insert' ), 10, 2 );
		add_filter( 'comment_text', array( $visitors, 'lets_review_comment_text' ), 10, 2 );
		add_action( 'comment_form_before_fields', array( $visitors, 'lets_review_form' ) );
		add_action( 'comment_form_logged_in_after', array( $visitors, 'lets_review_form' ) );
	}

	public function el_check() {
		echo '<span id="lr-check"></span>';
		$loader = Lets_Review_Frontend::asset_optimizer();
		if ( ! empty( $loader ) ) {
			include $this->dir_path . 'frontend/pswp.php';
		}
	}

	function thumbnails() {
		add_image_size( 'lets-review-360-240', 360, 240, true );
	}

	/**
	 * Let's Review Translation Loader
	 *
	 * @since 1.0.0
	 */
	public function lets_review_locale() {
		$i18n = new Lets_Review_i18n( $this->dir_path );
		add_action( 'plugins_loaded', array( $i18n, 'lets_review_textdomain' ) );
		add_filter( 'mce_external_languages', array( $i18n, 'lets_review_i18n_button' ) );
	}

	/**
	 * Let's Review Blocks
	 *
	 * @since 3.0.0
	 */
	public function lets_review_blocks() {
		$block = new Lets_Review_Blocks( $this->slug, $this->version, $this->url );
		add_action( 'enqueue_block_editor_assets', array( $block, 'lets_review_editor_assets' ) );
		add_filter( 'block_categories_all', array( $block, 'lets_review_block_category' ), 10, 2 );
		add_action( 'init', array( $block, 'lets_review_block_callbacks' ) );
	}
}
